#!/usr/bin/env bash

# Bash settings
set -o errexit    # abort on nonzero exitstatus
set -o pipefail   # don't hide errors within pipes
set -o nounset    # abort on unbound variable

# Variables
readonly script_name=$(basename "${0}")                                   # Name of the script
readonly script_dir=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )     # Directory of the script
IFS=$'\t\n'                                                               # Split on newlines and tabs (but not on spaces)
NCPU="${1:-4}"                                                            # Number of CPUs used by Julia

# Description of the script
echo -n "Running the script ${script_name} in ${script_dir}."
printf "\n"
echo -n "Each Julia job will use ${NCPU} CPUs."
printf "\n\n"

# Simple Bernoulli experiments with GK16
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all_GK16' --inst 'ber_triv' --K 5 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all_GK16" --inst "ber_triv" --K 5 --seed 42 --Nruns 5000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all_GK16' --inst 'ber_easy' --K 5 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all_GK16" --inst "ber_easy" --K 5 --seed 42 --Nruns 5000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all_GK16' --inst 'ber_eq3rd' --K 4 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all_GK16" --inst "ber_eq3rd" --K 4 --seed 42 --Nruns 5000
printf "\n\n"

# Simple Bernoulli experiments with TT
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all' --inst 'ber_triv' --K 5 --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all" --inst "ber_triv" --K 5 --seed 42 --Nruns 1000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all' --inst 'ber_easy' --K 5 --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all" --inst "ber_easy" --K 5 --seed 42 --Nruns 1000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'all' --inst 'ber_eq3rd' --K 4 --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "all" --inst "ber_eq3rd" --K 4 --seed 42 --Nruns 1000
printf "\n\n"

# Random experiments with GK16
for K in 8 10
do
  echo -n "Running: julia -O3 -p${NCPU} npbai_randoms.jl --expe 'random_GK16' --inst 'distinct' --K ${K} --seed 42 --Nruns 5000"
  printf "\n\n"
  julia -O3 -p"${NCPU}" npbai_randoms.jl --expe "random_GK16" --inst "distinct" --K "${K}" --seed 42 --Nruns 5000
  printf "\n\n"
done

# Random experiments with TT
echo -n "Running: julia -O3 -p${NCPU} npbai_randoms.jl --expe 'random' --inst 'distinct' --K 10 --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_randoms.jl --expe "random" --inst "distinct" --K 10 --seed 42 --Nruns 1000
printf "\n\n"

# DSSAT experiments with GK16
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'dsat_exp_GK16' --inst 'dsat_5' --seed 42 --Nruns 500"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "dsat_exp_GK16" --inst "dsat_5" --seed 42 --Nruns 500
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'dsat_exp_GK16' --inst 'dsat_6' --seed 42 --Nruns 250"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "dsat_exp_GK16" --inst "dsat_6" --seed 42 --Nruns 250
printf "\n\n"

# DSSAT experiments with TT
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'dsat_exp' --inst 'dsat_4' --seed 42 --Nruns 100"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "dsat_exp" --inst "dsat_4" --seed 42 --Nruns 100
printf "\n\n"

# Failure Equality Bernoulli experiments with GK16
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'fail_equ_GK16' --inst 'fail_eq2nd_easy' --K 3 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "fail_equ_GK16" --inst "fail_eq2nd_easy" --K 3 --seed 42 --Nruns 5000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'fail_equ_GK16' --inst 'fail_eq2nd_okay' --K 3 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "fail_equ_GK16" --inst "fail_eq2nd_okay" --K 3 --seed 42 --Nruns 5000
printf "\n\n"

echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'fail_equ_GK16' --inst 'fail_eq2nd_hard' --K 3 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "fail_equ_GK16" --inst "fail_eq2nd_hard" --K 3 --seed 42 --Nruns 5000
printf "\n\n"

# Failure Equality Bernoulli experiments with TT
echo -n "Running: julia -O3 -p${NCPU} npbai_experiments.jl --expe 'fail_equ' --inst 'fail_eq2nd_hard' --K 3 --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" npbai_experiments.jl --expe "fail_equ" --inst "fail_eq2nd_hard" --K 3 --seed 42 --Nruns 1000
printf "\n\n"
